/**
 * \file: GstreamerVideoSinkImpl.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <adit_logging.h>
#include "GstreamerVideoSinkImpl.h"
#include <uspi/EndpointDataSharing.h>

#include <inttypes.h>

LOG_IMPORT_CONTEXT(aauto_video)

using namespace std;

namespace adit { namespace aauto {

using uspi::SharedDataSender;

GstreamerVideoSinkImpl::GstreamerVideoSinkImpl(VideoSink* inSink, void* inSessionContext)
{
    videoSink = inSink;
    videoRunning = false;
    mShutdown = false;
    mIsSetThreadParam = false;
    mcallbacks = nullptr;
    gwidth = 0;
    gheight = 0;
    VideoPipeline = nullptr;
    sessionContext = inSessionContext;
    VideoSinkRecordRunning = false;
}

GstreamerVideoSinkImpl::~GstreamerVideoSinkImpl()
{
    if (!mShutdown)
    {
        shutdown();
    }
}


bool GstreamerVideoSinkImpl::init()
{
    bool ret = true;
    LOG_INFO((aauto_video, "VideoSink initialize"));

    VideoPipeline = new GstreamerVideoPipeline(mcallbacks, sessionContext);

    // Check configuration error
    if(!mConfig.ResultConfig())
    {
        LOG_ERROR((aauto_video, "Check configuration error"));
        ret = false;
    }

    workQueue = new WorkQueue(1, false, 0);
    if (workQueue != nullptr)
    {
        if (!workQueue->start())
        {
            LOG_ERROR((aauto_video, "error workQueue->start() in init()"));
            ret = false;
        }
    }

    if (mConfig.mCodecRes.size() > 0) {
        addSupportedConfigurations(MEDIA_CODEC_VIDEO_H264_BP);
    } else {
        addSupportedConfigurations(MEDIA_CODEC_VIDEO_H264_BP, mConfig.mWidth, mConfig.mHeight, mConfig.mDensity, mConfig.mPixelAspectRatio);
    }

    videoSink->registerCallbacks(this);
    videoSink->setMaxUnackedFrames(mConfig.mMaxUnackedFrames);

    mShutdown = false;
    return ret;
}

void GstreamerVideoSinkImpl::shutdown()
{
    mShutdown = true;

    /* Inform Application about stop */
    if (mcallbacks != nullptr) {
        mcallbacks->playbackStopCallback();
    }
    /* Stop pushing video frames into Gstreamer pipeline */
    if (workQueue != nullptr) {
        workQueue->shutdown();
        workQueue = nullptr;
    }
    /* Stop Gstreamer pipeline */
    if (VideoPipeline != nullptr) {
        if (videoRunning) {
            VideoPipeline->GstreamerStop();
            videoRunning = false;
        }

        delete VideoPipeline;
        VideoPipeline = nullptr;
    }

    if (VideoSinkRecordRunning) {
        VideoFrameRecord.close();
        VideoSinkRecordRunning = false;
    }

    LOGD_DEBUG((aauto_video, "VideoSink is down."));
    return;
}

void GstreamerVideoSinkImpl::setConfigItem(string inKey, string inValue)
{

    LOGD_DEBUG((aauto_video, "GstreamerVideoSinkImpl::setConfigItem inKey = %s"\
            ", inValue = %s", inKey.c_str(),inValue.c_str()));

    mConfig.set(inKey, inValue);
}

void GstreamerVideoSinkImpl::registerCallbacks(IAditVideoSinkCallbacks* inCallbacks)
{
    mcallbacks = inCallbacks;
}

void GstreamerVideoSinkImpl::dataAvailableCallback(int32_t inSessionId, uint64_t inTimestamp,
        uint8_t* inData, size_t inLen)
{
    (void)inSessionId;
    (void)inTimestamp;
    (void)inData;
    (void)inLen;

    LOG_ERROR((aauto_video, "Non zero copy version invoked. This is unsupported!"));
}

void GstreamerVideoSinkImpl::dataAvailableCallback(int32_t inSessionId, uint64_t inTimestamp,
        const shared_ptr<IoBuffer>& inFrame, uint8_t* inData, size_t inLen)
{
    if (workQueue != nullptr)
    {
        shared_ptr<WorkItem> frameItem(
                new FrameItem(*this, *VideoPipeline, inSessionId, inTimestamp, inFrame, inData, inLen));
        LOGD_VERBOSE((aauto_video, "queue video frame session=%d, len=%zu, inTimestamp=%" PRIu64 "", inSessionId, inLen, inTimestamp));

        // queue data for worker
        workQueue->queueWork(frameItem, 0);
    }
}

int GstreamerVideoSinkImpl::codecConfigCallback(uint8_t* inData, size_t inLen)
{
    // Sent in the beginning of every stream.
    LOGD_DEBUG((aauto_video, "codecConfig data length = %zu",inLen));

    if ((mConfig.mVideoRecord) && (VideoSinkRecordRunning)) {
        int res = VideoFrameRecord.write((char*)inData, inLen);
        if (res != (int)inLen) {
            LOG_ERROR((aauto_video, "VideoSink write data for VideoSinkRecord failed=%d", res));
        }
    }

    //the data in inData is copied to buffer for Gstreamer before ackFrame is called
    VideoPipeline->GstreamerPush(0, inData, inLen);

    return STATUS_SUCCESS;

}

int GstreamerVideoSinkImpl::setupCallback(int inMediaCodecType)
{
    /* This call signals that the other side wants
     * to set up a video stream and the information in this callback should be used to set
     * up any hardware necessary to handle the incoming stream type
     */
    LOGD_DEBUG((aauto_video, "%s() set up a video stream with codec type =%d", __FUNCTION__, inMediaCodecType));
    if(mcallbacks != nullptr)
    {
        mcallbacks->setupCallback(inMediaCodecType);
    }
    else
    {
        LOG_ERROR((aauto_video, "registercallback is not set."));
    }

    return STATUS_SUCCESS;
}

void GstreamerVideoSinkImpl::playbackStartCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_video, "start video playback, sessionId = %d", inSessionId));

    if (mConfig.mVideoRecord)
    {
        VideoFrameRecord.open(mConfig.mVideoFileRecord,"videosink");

        VideoSinkRecordRunning = true;
    }

    if (true == VideoPipeline->GstreamerStart(mConfig.mVideoPipeline, gwidth, gheight)) {
        LOGD_DEBUG((aauto_video, "GstreamerStart() done"));
        videoRunning = true;
    } else {
        LOG_ERROR((aauto_video, "GstreamerStart() failed!"));
        if (mcallbacks != nullptr)
        {
            mcallbacks->notifyErrorCallback(VIDEO_SINK_START_ERROR);
        }
    }

    if(mcallbacks != nullptr)
    {
        mcallbacks->playbackStartCallback();
    }
    else
    {
        LOG_ERROR((aauto_video, "registercallback is not set."));
    }
}

void GstreamerVideoSinkImpl::playbackStopCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_video, "stop video playback, sessionId = %d", inSessionId));

    videoRunning = false;

    if (mcallbacks != nullptr)
    {
        mcallbacks->playbackStopCallback();
    }
    else
    {
        LOG_ERROR((aauto_video, "registercallback is not set."));
    }

    VideoPipeline->GstreamerStop();

    if (VideoSinkRecordRunning) {
        VideoFrameRecord.close();
        VideoSinkRecordRunning = false;
    }
}

void GstreamerVideoSinkImpl::sourceVideoConfigCallback(int inIndex)
{
    LOGD_DEBUG((aauto_video, "video config index is %d ",inIndex));

    if ((size_t)inIndex >= mResolutions.size())
    {
        LOG_ERROR((aauto_video, "Source video config index is out of bounds."));

        if(mcallbacks != nullptr)
        {
            mcallbacks->notifyErrorCallback(VIDEO_SINK_CONFIGURATION_ERROR);
        }
    }
    else
    {
        mSelectedResolution = inIndex;

        PlayerResolution res;
        //get current resolution configuration from MD
        getCurrentResolution(&res);
        LOGD_DEBUG((aauto_video,"resolution from MD is %d x %d, width margin = %d"\
                " ,height margin = %d" , res.w, res.h, res.tlx, res.tly));

        SharedDataSender::instance().transmitResolutionData(sessionContext, res.w, res.h, res.tlx, res.tly);

        //set resolution for Gstreamer
        gwidth = res.w;
        gheight = res.h;
        if(mcallbacks != nullptr)
        {
            mcallbacks->sourceVideoConfigCallback(res.w, res.h, res.w - res.tlx*2, res.h - res.tly*2);
        }
        else
        {
            LOG_ERROR((aauto_video, "registercallback is not set."));
        }
    }
}

void GstreamerVideoSinkImpl::videoFocusCallback(int inFocus, int inReason)
{
    (void)inReason;

    LOGD_DEBUG((aauto_video, "video focus callback focus=%d, reason=%d", inFocus, inReason));

    if(mcallbacks != nullptr)
    {
        mcallbacks->videoFocusCallback(inFocus, inReason);
    }
    else
    {
        LOG_ERROR((aauto_video, "registercallback is not set."));
    }
}

void GstreamerVideoSinkImpl::FrameItem::SetThreadParam(const char* threadName)
{
    /* set thread name */
    string sName = threadName; //too long name, less than 15 chars ??
    prctl(PR_SET_NAME, sName.c_str(), 0, 0, 0);

    if(!sink.mConfig.mDisablePrio)
    {
        /* set thread priority */
        int err = 0;
        int schedPolicy;
        struct sched_param schedParam;

        err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
        if(err == 0)
        {
            schedParam.sched_priority = sink.mConfig.mThreadPrio;
            err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
            if(err != 0)
            {
                LOG_ERROR((aauto_video, "%s: set priority failed with error %d", sName.c_str(), err));
            }
        }
        else
        {
            LOG_ERROR((aauto_video, "%s: get priority failed with error %d", sName.c_str(), err));
        }
    }
}

void GstreamerVideoSinkImpl::FrameItem::run()
{
    if (!sink.mIsSetThreadParam)
    {
        // set thread name
        SetThreadParam("VideoSinkWorkQueue");
        sink.mIsSetThreadParam = true; /* setting only once */
    }
    if((sink.mConfig.mVideoRecord) && (sink.VideoSinkRecordRunning)){
        int res = sink.VideoFrameRecord.write((char*)mDataPtr,mLen);
        if (res != (int)mLen) {
            LOG_ERROR((aauto_video, "VideoSink write data for VideoSinkRecord failed=%d", res));
        }
    }
    Video.GstreamerPush(mTimestamp, mDataPtr, mLen);
    //ackFrame for VideoSink
    sink.videoSink->ackFrames(mSessionId, 1);
}


void GstreamerVideoSinkImpl::calculateClipped(int x, int y, int cx, int cy, PlayerResolution* res)
{
    int widthMargin = (cx - x) / 2;
    int heightMargin = (cy - y) / 2;
    res->w = cx;
    res->h = cy;
    res->tlx = widthMargin;
    res->tly = heightMargin;
    res->brx = cx - widthMargin;
    res->bry = cy - heightMargin;
}

void GstreamerVideoSinkImpl::addSupportedConfigurations(int32_t codec, int32_t inWidth, int32_t inHeigth, int32_t inDensity, float inPixelAspectRatio)
{
    LOG_WARN((aauto_video, "Configuration values are deprecated! Please use video-codec-resolution-800x480/-1280x720/-1920x1080"));
    LOG_WARN((aauto_video, "  example:  'video-codec-resolution-800x480      width=800 height=480 density=160 pixelAspectRatio=1.0 additional-depth=0 fps=30'"));

    /** set mandatory resolution shall be not necessary - begin**/

    float _ratio = inPixelAspectRatio * 1e4;

    /* First, add the mandatory resolution 800x480 */
    /* set the ratio of the physical pixel width over height multiplied by 1e4 (10000) */
    PlayerResolution res = {0, 0, 800, 480, 800, 480, 0, (int)(_ratio + 0.5f), 0, 0};

    /* calculate the density of the mandatory resolution (800x480) */
    float fHigh = (static_cast<float>(res.h) / static_cast<float>(inHeigth));
    res.density = static_cast<int>(fHigh * inDensity);

    if ((inWidth > 800) || (inHeigth > 480)) {
        /* insert the mandatory resolution (800x480) */
        mResolutions.push_back(res);
    }
    /** set mandatory resolution shall be not necessary - end **/

    memset(&res, 0, sizeof(PlayerResolution));

    /* mandatory frame rate = 30 fps */
    int fps = VIDEO_FPS_30;
    if (mConfig.mFps == 30) {
        fps = VIDEO_FPS_30;
    } else if(mConfig.mFps == 60) {
        fps = VIDEO_FPS_60;
    } else {
        LOG_WARN((aauto_video, "%s()  unsupported fps=%d. Use mandatory fps=%d", __FUNCTION__, mConfig.mFps, 30));
        fps = VIDEO_FPS_30;
    }

    /* Set the codec type */
    videoSink->setCodecType(codec);

    /* set the density for the configured resolution */
    res.density = inDensity;
    /* set the ratio of the physical pixel width over height for the configured resolution */
    _ratio = inPixelAspectRatio * 1e4;
    res.pixelAspectRatio = (int)(_ratio + 0.5f);

    /* number of elements in VideoCodecResolutions[] depends on the supported resolutions.
     * see enum VideoCodecResolutionType*/
    VideoResolution VideoCodecResolutions[] = {{800,480}, {1280, 720}, {1920, 1080}};
    for (unsigned int i = 0; i < (sizeof(VideoCodecResolutions) / sizeof(VideoCodecResolutions[0])); i++)
    {
        if ((inWidth <= VideoCodecResolutions[i].width) && (inHeigth <= VideoCodecResolutions[i].height)) {
            calculateClipped(inWidth, inHeigth, VideoCodecResolutions[i].width, VideoCodecResolutions[i].height, &res);

            /* During AAP service discovery, MD request a list of VideoConfigurations.
             * HU send a prioritized list of indices (most preferred first) to MD.
             * MD selects a configuration based on HU preference. */
            mResolutions.push_front(res);
            break; // leave
        }
    }

    /* Send supported resolution with margin to MD */
    /* resolution (width, height) mapped by VideoCodecResolutionType */
    for (deque<PlayerResolution>::const_iterator it = mResolutions.begin();
            it != mResolutions.end(); ++it)
    {
        int codecResolution = 0;

        switch(it->w)
        {
            case 800:
                codecResolution = VideoCodecResolutionType::VIDEO_800x480;
                break;
            case 1280:
                codecResolution = VideoCodecResolutionType::VIDEO_1280x720;
                break;
            case 1920:
                codecResolution = VideoCodecResolutionType::VIDEO_1920x1080;
                break;
        } // switch
        /*
         * @param codecResolution This can be one of VIDEO_800x480, VIDEO_1280x720 or VIDEO_1920x1080.
         * @param fps frame rate of video. Either 60 or 30.
         * @param widthMargin The number of pixels in the x axis that content wont be rendered to.
         * @param heightMargin The number of pixels in the y axis that content wont be rendered to.
         * ...
         */
        /* VideoSink::addSupportedConfiguration(codecResolution, fps, widthMargin, heightMargin, ...) */
        videoSink->addSupportedConfiguration(codecResolution,           // codecResolution This can be one of VIDEO_800x480, VIDEO_1280x720 or VIDEO_1920x1080
                                             fps,                       // frame rate of video. Either 60 or 30.
                                             (it->w - it->brx) * 2,     // widthMargin The number of pixels in the x axis that content wont be rendered to
                                             (it->h - it->bry) * 2,     // heightMargin The number of pixels in the y axis that content wont be rendered to.
                                             it->density,
                                             mConfig.mAdditionalDepth,
                                             it->pixelAspectRatio);

        LOGD_DEBUG((aauto_video,"addSupportedConfiguration width=%d height=%d widthMargin=%d heightMargin=%d fps=%d density=%d additionalDepth=%d pixelAspectRatio=%d",
                it->w, it->h, (it->w - it->brx), (it->h - it->bry), fps, it->density, mConfig.mAdditionalDepth, it->pixelAspectRatio));
    } // for
}

void GstreamerVideoSinkImpl::addSupportedConfigurations(int32_t codec)
{
    LOGD_DEBUG((aauto_video, "%s()  Configured VideoCodecResolutions: %zu", __func__, mConfig.mCodecRes.size()));

    float _ratio = 0;
    PlayerResolution res = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    /* Set the codec type */
    videoSink->setCodecType(codec);

    for (deque<VideoCodecRes>::const_iterator iter = mConfig.mCodecRes.begin(); iter != mConfig.mCodecRes.end(); ++iter)
    {
        memset(&res, 0, sizeof(PlayerResolution));

        /* mandatory frame rate = 30 fps */
        if (iter->fps == 30) {
            res.fps = VIDEO_FPS_30;
        } else if(iter->fps == 60) {
            res.fps = VIDEO_FPS_60;
        } else {
            LOG_WARN((aauto_video, "%s()  unsupported fps=%d. Use mandatory fps=%d", __func__, iter->fps, 30));
            res.fps = VIDEO_FPS_30;
        }

        /* set the density for the configured resolution */
        res.density = iter->density;

        /* set the ratio of the physical pixel width over height multiplied by 1e4 (10000) */
        _ratio = iter->pixelAspectRatio * 1e4;
        res.pixelAspectRatio = (int)(_ratio + 0.5f);

        res.additionalDepth = iter->additionalDepth;

        /* number of elements in VideoCodecResolutions[] depends on the supported resolutions.
         * see enum VideoCodecResolutionType*/
        VideoResolution VideoCodecResolutions[] = {{800,480}, {1280, 720}, {1920, 1080}};
        for (unsigned int i = 0; i < (sizeof(VideoCodecResolutions) / sizeof(VideoCodecResolutions[0])); i++)
        {
            if ((iter->width <= VideoCodecResolutions[i].width) && (iter->height <= VideoCodecResolutions[i].height)) {
                calculateClipped(iter->width, iter->height, VideoCodecResolutions[i].width, VideoCodecResolutions[i].height, &res);

                /* During AAP service discovery, MD request a list of VideoConfigurations.
                 * HU send a prioritized list of indices (most preferred first) to MD.
                 * MD selects a configuration based on HU preference. */
                mResolutions.push_back(res);
                break; // leave
            }
        }

        /* Send supported resolution with margin to MD */
        /* resolution (width, height) mapped by VideoCodecResolutionType */
        int codecResolution = 0;
        switch(res.w)
        {
            case 800:
                codecResolution = VideoCodecResolutionType::VIDEO_800x480;
                break;
            case 1280:
                codecResolution = VideoCodecResolutionType::VIDEO_1280x720;
                break;
            case 1920:
                codecResolution = VideoCodecResolutionType::VIDEO_1920x1080;
                break;
            default:
                LOG_WARN((aauto_video, "%s()  Could not set VideoCodecResolution for width=%d", __func__, res.w));
                break;
        } // switch

        /*
         * @param codecResolution This can be one of VIDEO_800x480, VIDEO_1280x720 or VIDEO_1920x1080.
         * @param fps frame rate of video. Either 60 or 30.
         * @param widthMargin The number of pixels in the x axis that content wont be rendered to.
         * @param heightMargin The number of pixels in the y axis that content wont be rendered to.
         * ...
         */
        /* VideoSink::addSupportedConfiguration(codecResolution, fps, widthMargin, heightMargin, ...) */
        int indexNum = videoSink->addSupportedConfiguration(codecResolution,           // codecResolution This can be one of VIDEO_800x480, VIDEO_1280x720 or VIDEO_1920x1080
                                                            res.fps,                   // frame rate of video. Either 60 or 30.
                                                            (res.w - res.brx) * 2,     // widthMargin The number of pixels in the x axis that content wont be rendered to
                                                            (res.h - res.bry) * 2,     // heightMargin The number of pixels in the y axis that content wont be rendered to.
                                                            res.density,
                                                            res.additionalDepth,
                                                            res.pixelAspectRatio);

        LOGD_DEBUG((aauto_video,"%s()  added codecRes=%d width=%d height=%d wMargin=%d hMargin=%d fps=%d density=%d addDepth=%d pixelAspectRatio=%d with indexNum=%d",
                __func__, codecResolution, res.w, res.h, (res.w - res.brx), (res.h - res.bry), res.fps, res.density, res.additionalDepth, res.pixelAspectRatio, indexNum));
    } // for
}

bool GstreamerVideoSinkImpl::getCurrentResolution(PlayerResolution* res)
{
    if (mSelectedResolution == -1)
    {
        return false;
    }
    const PlayerResolution& selected = mResolutions[mSelectedResolution];
    res->tlx = selected.tlx;
    res->tly = selected.tly;
    res->brx = selected.brx;
    res->bry = selected.bry;
    res->w = selected.w;
    res->h = selected.h;
    res->density = selected.density;
    return true;
}

} } // namespace adit { namespace aauto {
